/**
 * Use this file for JavaScript code that you want to run in the front-end
 * on posts/pages that contain this block.
 *
 * When this file is defined as the value of the `viewScript` property
 * in `block.json` it will be enqueued on the front end of the site.
 *
 * Example:
 *
 * ```js
 * {
 *   "viewScript": "file:./view.js"
 * }
 * ```
 *
 * If you're not making any changes to this file because your project doesn't need any
 * JavaScript running in the front-end, then you should delete this file and remove
 * the `viewScript` property from `block.json`.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-metadata/#view-script
 */

document.addEventListener('DOMContentLoaded', function () {
	// ページ上のすべての my-expander を対象とする
	var items = document.querySelectorAll('.my-expander');

	function updateAncestors(expander, extraHeight) {
		var parent = expander.parentElement;

		while (parent) {
			if (parent.classList && parent.classList.contains('my-expander')) {
				var parentContent = parent.querySelector(':scope > .my-expander-content');
				if (parentContent && parent.classList.contains('is-open')) {
					var maxHeight = parentContent.scrollHeight + extraHeight;
					parentContent.style.maxHeight = maxHeight + 'px';
				}
			}
			parent = parent.parentElement;
		}
	}

	function triggerRelayout() {
		// レイアウト変更が反映された直後に発火させる
		requestAnimationFrame(function () {
			// resize イベントでブラウザとテーマのレイアウトを更新
			window.dispatchEvent(new Event('resize'));
		});
	}

	items.forEach(function (item) {
		var title = item.querySelector('.my-expander-title');
		var content = item.querySelector(':scope > .my-expander-content');

		if (!title || !content) return;

		// 初期状態は閉じる
		item.classList.remove('is-open');
		content.style.maxHeight = '0px';

		// max-height アニメーションが終わったら親の高さを更新
		content.addEventListener('transitionend', function (e) {
			if (e.propertyName !== 'max-height') return;
			if (!item.classList.contains('is-open')) {
				updateAncestors(item, 0);
				triggerRelayout();
			}
		});

		// タイトルがクリックされたときの処理
		title.addEventListener('click', function () {
			var isOpen = item.classList.contains('is-open');

			if (isOpen) {
				// ---- 自分を閉じる ----
				item.classList.remove('is-open');
				content.style.maxHeight = '0px';
			} else {
				// ---- 自分を開く ----
				item.classList.add('is-open');

				// 中身の高さを取得して、その分だけmax-heightを広げる
				var fullHeight = content.scrollHeight; // px 単位の数値
				content.style.maxHeight = fullHeight + 'px';
				updateAncestors(item, fullHeight);
			}
		});
	});
});
