/**
 * Retrieves the translation of text.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-i18n/
 */
import { __ } from '@wordpress/i18n';

/**
 * React hook that is used to mark the block wrapper element.
 * It provides all the necessary props like the class name.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/packages/packages-block-editor/#useblockprops
 */
import { useBlockProps, RichText, InnerBlocks, InspectorControls } from '@wordpress/block-editor';
import { useState } from '@wordpress/element';
import { PanelBody, ToggleControl, RangeControl, ColorPalette } from '@wordpress/components';

/**
 * Lets webpack process CSS, SASS or SCSS files referenced in JavaScript files.
 * Those files can contain any CSS code that gets applied to the editor.
 *
 * @see https://www.npmjs.com/package/@wordpress/scripts#using-css
 */
import './editor.scss';

const COLORS = [
	{ name: '薄いグレー', color: '#dddddd' },
	{ name: '黒', color: '#000000' },
	{ name: '青', color: '#0073aa' },
	{ name: 'オレンジ', color: '#ff9900' },
];

/**
 * The edit function describes the structure of your block in the context of the
 * editor. This represents what the editor will render when the block is used.
 *
 * @see https://developer.wordpress.org/block-editor/reference-guides/block-api/block-edit-save/#edit
 *
 * @return {Element} Element to render.
 */
export default function Edit({ attributes, setAttributes }) {
	const { title, borderEnabled, borderColor, borderWidth } = attributes;
	const [isOpen, setIsOpen] = useState(true);
	// 枠線スタイル（有効なときだけ付ける）
	const style =
		borderEnabled
			? {
				border: `${borderWidth || 1}px solid ${borderColor || '#dddddd'}`,
				borderRadius: '4px',
			}
			: { border: 'none' };
	const blockProps = useBlockProps({
		className: `my-expander${isOpen ? ' is-open' : ''}`,
		style,
	});

	const onToggle = () => {
		setIsOpen((prev) => !prev);
	};

	// エディタ用：ここで表示／非表示を制御
	const contentStyle = isOpen
		? { maxHeight: 'none', overflow: 'visible', display: 'block' }
		: { maxHeight: 0, overflow: 'hidden', display: 'none' };

	return (
		<>
			<InspectorControls>
				<PanelBody title="枠線の設定" initialOpen={true}>
					<ToggleControl
						label="枠線を表示する"
						checked={borderEnabled}
						onChange={(value) => setAttributes({ borderEnabled: value })}
					/>
					{borderEnabled && (
						<>
							<p>枠線の色</p>
							<ColorPalette
								colors={COLORS}
								value={borderColor}
								onChange={(color) => setAttributes({ borderColor: color })}
							/>
							<RangeControl
								label="枠線の太さ（px）"
								value={borderWidth}
								onChange={(value) => setAttributes({ borderWidth: value })}
								min={0}
								max={10}
							/>
						</>
					)}
				</PanelBody>
			</InspectorControls>

			<div {...blockProps}>
				<div className="my-expander-title" onClick={onToggle}>
					<RichText
						tagName="span"
						value={title}
						onChange={(value) => setAttributes({ title: value })}
						placeholder={__('タイトルを入力…', 'my-expander-block')}
					/>
				</div>
				<div className="my-expander-content" style={contentStyle}>
					<InnerBlocks
						placeholder={__('ここにブロックを追加…', 'my-expander-block')}
					// allowedBlocks で許可するブロックを絞ることも可能
					// allowedBlocks={ [ 'core/paragraph', 'core/image' ] }
					/>
				</div>
			</div>
		</>
	);
}
