<?php

declare(strict_types=1);

enum HttpMethod: int
{
    case UNKNOWN = 0;
    case GET = 1;
    case POST = 2;
    case PUT = 3;
    case DELETE = 4;

    public static function tryParse(string $text): HttpMethod
    {
        switch ($text) {
            case HttpMethod::GET->name:
                return HttpMethod::GET;
            case HttpMethod::POST->name:
                return HttpMethod::POST;
            case HttpMethod::PUT->name:
                return HttpMethod::PUT;
            case HttpMethod::DELETE->name:
                return HttpMethod::DELETE;
            default:
                return HttpMethod::UNKNOWN;
        }
    }
}

/**
 * @return array<string>
 */
function get_params(string $key, string $default_val, bool $is_post = true): array
{
    $src = $is_post ? $_POST : $_GET;
    $arr = $src[$key];
    return toStrArray($arr);
}

/**
 * @return array<string, string|int|float>
 */
function get_asoc_params(string $key, string $default_val, bool $is_post = true): array
{
    $src = $is_post ? $_POST : $_GET;
    $arr = $src[$key];
    return toAssocArray($arr);
}

function get_param(string $key, string $default_val, bool $is_post = true): string
{
    $src = $is_post ? $_POST : $_GET;
    $val = $src[$key];
    return is_string($val) ? toString($val) : $default_val;
}

function get_item(mixed $array, string $key): string
{
    $value =  (is_array($array) && array_key_exists($key, $array)) ? toString($array[$key]) : '';
    return htmlspecialchars($value);
}

function redirect(string $path): void
{
    if ($path === GO_HOME) {
        $path = get_url('');
    } else if ($path === GO_REFERER) {
        $path = toString($_SERVER['HTTP_REFERER']);
    } else {
        $path = get_url($path);
    }

    header("Location: {$path}");

    die();
}

function get_url(string $path): string
{
    return BASE_CONTEXT_PATH . trim($path, '/');
}

/**
 * @return array{method: HttpMethod, path: string, query: string}
 */
function get_method(): array
{
    $req_method = strtoupper(toString($_SERVER['REQUEST_METHOD']));
    $url = parse_url(toString(CURRENT_URI));
    if (mb_strlen(BASE_CONTEXT_PATH) > 1) {
        // @phpstan-ignore offsetAccess.nonOffsetAccessible
        $path = ltrim(rtrim(str_replace(BASE_CONTEXT_PATH, '', $url['path']), '/'), '/');
    } else {
        // @phpstan-ignore offsetAccess.nonOffsetAccessible
        $path = ltrim(rtrim($url['path'], '/'), '/');
    }
    $query = $url['query'] ?? '';

    $match = [];
    $f = fn($a) => preg_match('/method=(put|delete)$/i', toString($a), $match) ? strtoupper($match[1]) : '';
    $items = array_reverse(explode('?', $query));
    $items = array_values(array_diff(array_map($f, $items), ['']));

    if (count($items) > 0) {
        $method = HttpMethod::tryParse($items[0]);
    } else {
        $method = HttpMethod::tryParse($req_method);
    }

    return compact('method', 'path', 'query');
}

function forwardGeocode(string $location): string
{
    // @phpstan-ignore empty.expr
    if (empty(MAPBOX_TOKEN)) return '';

    $mapbox_api = 'https://api.mapbox.com/search/geocode/v6/forward';
    $params = array(
        'q' => $location,
        'proximity' => 'ip',
        'access_token' => MAPBOX_TOKEN
    );
    $query = http_build_query($params);
    $url = "{$mapbox_api}?{$query}";

    if (IS_WINDOWS)
        $response = forwardGeocodeWin($url);
    else
        $response = forwardGeocodeLinux($url);

    if (empty($response)) return '';
    $result = json_decode($response, true);
    if (empty($result) || !is_array($result['features']) || count($result['features']) === 0) return '';
    $geometry = json_encode($result['features'][0]['geometry'], JSON_UNESCAPED_UNICODE);

    return $geometry ? $geometry : '';
}

/**
 * @param non-empty-string $url
 */
function forwardGeocodeLinux(string $url): string
{
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);

    return is_string($response) ? $response : '';
}

function forwardGeocodeWin(string $url): string
{
    $context = stream_context_create(['http' => ['timeout' => 5]]);
    $response = @file_get_contents($url, FALSE, $context);

    return is_string($response) ? $response : '';
}

/**
 * @param array<models\CampGroundSchema> $campgrounds
 * @return array<array<string, mixed>>
 */
function getGeoDatas(array $campgrounds): array
{
    $datas = [];

    foreach ($campgrounds as $camp)
        array_push($datas, $camp->getGeoData());

    return $datas;
}

function toObject(mixed $arg): ?object
{
    return is_object($arg) ? (object)$arg : null;
}

function toString(mixed $arg): string
{
    return is_string($arg) ? strval($arg) : '';
}

/**
 * @return array<string>
 */
function toStrArray(mixed $arg): array
{
    $f = fn($a): string => is_string($a) ? strval($a) : '';
    $arr = [];
    if (is_array($arg)) $arr = (array)$arg;
    $items = array_values(array_diff(array_map($f, $arr), ['']));

    return $items;
}

/**
 * @return array<string, string|int|float>
 */
function toAssocArray(mixed $arg): array
{
    if (!is_array($arg)) return [];

    $results = [];
    $arr = (array)$arg;

    foreach ($arr as $key => $value) {
        if (!is_string($key) || (!is_string($value) && !is_numeric($value))) continue;
        $results[$key] = $value;
    }

    return $results;
}

/**
 * @return array<array<string, string|int|float>>
 */
function toAssocArrArray(mixed $arg): array
{
    if (!is_array($arg)) return [];

    $results = [];
    $arr = (array)$arg;

    foreach ($arr as $item) {
        $child_arr = toAssocArray($item);
        if (!empty($child_arr)) array_push($results, $child_arr);
    }

    return $results;
}
