<?php

declare(strict_types=1);

require_once '../vendor/autoload.php';

require_once '../config.php';
require_once '../db/datasource.php';
require_once '../db/users.query.php';
require_once '../db/images.query.php';
require_once '../db/reviews.query.php';
require_once '../db/campgrounds.query.php';
require_once '../utils/helper.php';
require_once '../utils/session.php';
require_once '../models/user.php';
require_once '../models/images.php';
require_once '../models/campground.php';

require_once 'cities.php';
require_once 'seedHelpers.php';

use db\UserQuery;
use db\ImageQuery;
use db\ReviewQuery;
use db\CampGroundQuery;
use models\UserSchema;
use models\CampGroundSchema;

// SSE の基本設定
header("Content-Type: text/event-stream");
header("Cache-Control: no-cache");
header("Connection: keep-alive");

// 出力バッファを制御
while (ob_get_level()) ob_end_flush();
flush();

function sendProgress(int $percent, string $message): void
{
    echo "data: {$percent},{$message}\n\n";
    flush();
}

$seedDB = function () {

    // ユーザー 'test' が存在しなければ作成
    $user = UserQuery::fetchByUserName('test');
    if ($user === null) {
        $new = new UserSchema;
        $new->username = 'test';
        $new->email    = 'test@test';
        $new->password = 'test';

        if (UserQuery::insert($new)) {
            sendProgress(0, "ユーザー 'test' を作成しました。");
        } else {
            sendProgress(0, "ユーザー 'test' の作成に失敗しました。");
        }
    } else {
        sendProgress(0, "ユーザー 'test' は既に存在します。");
    }

    sendProgress(0, 'データ生成を開始します...');

    global $cities;
    global $descriptors;
    global $places;

    ImageQuery::deleteAll();
    ReviewQuery::deleteAll();
    CampGroundQuery::deleteAll();

    sendProgress(0, '既存データを削除しました。');

    $sample1 = fn(array $array): string => toString($array[rand(0, count($array) - 1)]);
    $sample2 = fn(array $array): array => toAssocArray($array[rand(0, count($array) - 1)]);

    sendProgress(0, '生成を開始します...');

    $total = 50;

    for ($i = 1; $i <= $total; $i++) {

        $title = $sample1(toStrArray($descriptors)) . '・' . $sample1(toStrArray($places));
        $data = $sample2(toAssocArrArray($cities));
        ['prefecture' => $pref, 'city' => $city] = $data;
        $location = $pref . $city;
        $price = rand(1000, 3000);
        $camp = new CampGroundSchema;
        $camp->author = 'test';
        $camp->title = $title;
        $camp->location = $location;
        $camp->geometry = forwardGeocode($location);
        $image1 = 'assets/' . sprintf('%03d', rand(1, 101)) . '.png';
        $image2 = 'assets/' . sprintf('%03d', rand(1, 101)) . '.png';
        $images = [];
        if (file_exists('../' . $image1)) array_push($images, $image1);
        if (file_exists('../' . $image2)) array_push($images, $image2);
        $camp->addImages($images);
        $camp->description = '木曾路はすべて山の中である。あるところは岨づたいに行く崖の道であり、あるところは数十間の深さに臨む木曾川の岸であり、あるところは山の尾をめぐる谷の入り口である。一筋の街道はこの深い森林地帯を貫いていた。東ざかいの桜沢から、西の十曲峠まで、木曾十一宿はこの街道に添うて、二十二里余にわたる長い谿谷の間に散在していた。道路の位置も幾たびか改まったもので、古道はいつのまにか深い山間に埋もれた。';
        $camp->price = $price;

        $percent = intval($i / $total * 100);
        sendProgress($percent, "データ生成中... {$i} / {$total}");

        CampGroundQuery::save($camp);
    }

    sendProgress(100, "完了");
    echo "data: DONE\n\n";
    flush();
};

$seedDB();
