<?php

declare(strict_types=1);

/**
 * MySQLデータベースをPDOで操作するための基本クラス
 */

namespace db;

use PDO;
use PDOStatement;
use Throwable;

/**
 * PDOオブジェクトのシングルトンクラス定義
 */
class PDOSingleton
{
    private static PDO $singleton;
    private PDO $conn;

    /**
     * コンストラクタ
     */
    private function __construct(string $dsn, string $username, string $password)
    {
        $this->conn = new PDO($dsn, $username, $password);
        $this->conn->setAttribute(PDO::ATTR_DEFAULT_FETCH_MODE, PDO::FETCH_ASSOC);
        $this->conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        $this->conn->setAttribute(PDO::ATTR_EMULATE_PREPARES, false);
    }

    /**
     * 単一のPDOオブジェクトを返します
     */
    public static function getInstance(string $dsn, string $username, string $password): PDO
    {
        if (!isset(self::$singleton)) {
            $instance = new PDOSingleton($dsn, $username, $password);
            self::$singleton = $instance->conn;
        }
        return self::$singleton;
    }
}

/**
 * MySQL接続クラス定義
 */
class DataSource
{
    private PDO $conn;
    private bool $sqlResult;
    public const CLS = 'cls';

    /**
     * コンストラクタ
     */
    public function __construct(string $host = 'localhost', string $port = MYSQL_PORT, string $dbName = 'YelpCampCleanArch', string $username = 'clean_arch', string $password = 'password')
    {
        try {
            $dsn = "mysql:host={$host};port={$port};dbname={$dbName};";
            $this->conn = PDOSingleton::getInstance($dsn, $username, $password);
        } catch (Throwable $e) {
            die('Datasource error.');
        }
    }

    /**
     * @param array<string, int|string> $params
     * @return array<object>
     */
    public function select(string $sql = "", $params = [], string $type = '', string $cls = ''): array
    {
        $stmt = $this->executeSql($sql, $params);

        if (!$stmt) return [];
        $results = [];
        $selected = [];

        if ($type === static::CLS) {
            $selected = $stmt->fetchAll(PDO::FETCH_CLASS, $cls);
        } else {
            $selected = $stmt->fetchAll();
        }

        if (!empty($selected)) {
            foreach ($selected as $item) {
                $obj = toObject($item);
                if (!empty($obj)) array_push($results, $obj);
            }
        }

        return $results;
    }

    /**
     * @param array<string, int|string> $params
     */
    public function execute(string $sql = "", $params = []): bool
    {
        $this->executeSql($sql, $params);
        return  $this->sqlResult;
    }

    /**
     * @param array<string, int|string> $params
     */
    public function selectOne(string $sql = "", $params = [], string $type = '', string $cls = ''): ?object
    {
        $result = $this->select($sql, $params, $type, $cls);
        return count($result) > 0 ? $result[0] : null;
    }

    public function begin(): void
    {
        $this->conn->beginTransaction();
    }

    public function commit(): void
    {
        $this->conn->commit();
    }

    public function rollback(): void
    {
        $this->conn->rollback();
    }

    /**
     * @param array<string, int|string> $params
     */
    private function executeSql(string $sql, array $params): PDOStatement|false
    {
        $stmt = $this->conn->prepare($sql);
        $this->sqlResult = $stmt->execute($params);
        return $stmt;
    }

    public function lastInsertId(): string|false
    {
        return $this->conn->lastInsertId();
    }
}
