<?php

declare(strict_types=1);

namespace db;

use db\DataSource;
use models\CampGroundSchema;

class CampGroundQuery
{
    public static function findByIdAndDelete(int $id): bool
    {
        $db = new DataSource;

        try {
            $success = false;
            $db->begin();

            if (!ImageQuery::findByCampIdAndDelete($id)) return false;
            if (!ReviewQuery::findByCampIdAndDelete($id)) return false;
            $sql = 'delete from campgrounds where id = :id;';
            $success = $db->execute($sql, [':id' => $id]);

            return $success;
        } finally {
            if ($success)
                $db->commit();
            else
                $db->rollback();
        }
    }

    public static function update(CampGroundSchema $campground): ?CampGroundSchema
    {
        $db = new DataSource;

        try {
            $success = false;
            $db->begin();
            $sql = 'update campgrounds set title = :title, price = :price, description = :description, location = :location, author = :author, geometry = :geometry where id = :id';

            $result = $db->execute($sql, [
                ':title' => $campground->title,
                ':price' => $campground->price,
                ':description' => $campground->description,
                ':location' => $campground->location,
                ':author' => $campground->author,
                ':geometry' => $campground->geometry,
                ':id' => $campground->id,
            ]);
            if (!$result) return null;
            if (!ImageQuery::save($campground->id, $campground->images, exclude_exist: true)) return null;
            if (!ImageQuery::deleteImages($campground->id, $campground->deleteImages)) return null;

            $success = true;

            return $campground;
        } finally {
            if ($success)
                $db->commit();
            else
                $db->rollback();
        }
    }

    public static function save(CampGroundSchema $campground): ?CampGroundSchema
    {
        $db = new DataSource;

        try {
            $success = false;
            $db->begin();
            if (!self::insert($campground)) return null;
            $id = $db->lastInsertId();
            $campground->id = intval($id);
            ImageQuery::save($campground->id, $campground->images);

            $success = true;

            return $campground;
        } finally {
            if ($success)
                $db->commit();
            else
                $db->rollback();
        }
    }

    private static function insert(CampGroundSchema $campground): bool
    {
        $db = new DataSource;
        $sql = 'insert into campgrounds (title, price, description, location, author, geometry) values (:title, :price, :description, :location, :author, :geometry)';

        return $db->execute($sql, [
            ':title' => $campground->title,
            ':price' => $campground->price,
            ':description' => $campground->description,
            ':location' => $campground->location,
            ':author' => $campground->author,
            ':geometry' => $campground->geometry,
        ]);
    }

    public static function deleteAll(): bool
    {
        $db  = new DataSource;
        $sql = 'delete from campgrounds';

        return $db->execute($sql);
    }

    /**
     * @return array<int, CampGroundSchema>
     */
    public static function select(?string $criteria = null): array
    {
        $db = new DataSource;
        $sql = 'select * from campgrounds ' . (empty($criteria) ? '' : " where {$criteria}");
        $selected = $db->select($sql, [], DataSource::CLS, CampGroundSchema::class);
        $results =[];
        if (!empty($selected)) {
            foreach ($selected as $item) {
                $camp = CampGroundSchema::cast($item);
                $camp->images = ImageQuery::select("camp_id = {$camp->id}");
                array_push($results, $camp);
            }
        }

        return $results;
    }

    public static function findById(int $id): ?CampGroundSchema
    {
        $db = new DataSource;
        $sql = 'select * from campgrounds where id = :id';
        $selected = $db->select($sql, [':id' => $id,], DataSource::CLS, CampGroundSchema::class);
        if (empty($selected)) return null;
        $camp = CampGroundSchema::cast($selected[0]);
        $camp->images = ImageQuery::select("camp_id = {$camp->id}");
        return $camp;
    }
}
