<?php

use Domain\Entities\User;
use Domain\Repositories\UserRepositoryInterface;
use Application\UseCases\User\UserRegisterInput;
use Application\UseCases\User\UserRegisterOutput;
use Application\UseCases\User\UserRegisterUseCase;
use Application\Exception\ApplicationException;

it('registers a new user successfully', function () {

    // -------- Mock Repository --------
    $repo = $this->createMock(UserRepositoryInterface::class);

    // findByUsername → null（同名ユーザーなし）
    $repo->method('findByUsername')
        ->willReturn(null);

    // save() されたら、User を返す
    $repo->method('save')
        ->willReturnCallback(function (User $user) {
            return new User(
                username: $user->username(),
                email: $user->email(),
                passwordHash: $user->passwordHash(),
            );
        });

    // -------- UseCase --------
    $useCase = new UserRegisterUseCase($repo);

    $input = new UserRegisterInput(
        username: 'alice',
        email: 'alice@example.com',
        plainPassword: 'secret123'
    );

    $output = $useCase->execute($input);

    // -------- Assertions --------
    expect($output)->toBeInstanceOf(UserRegisterOutput::class);
    expect($output->username())->toBe('alice');
});

it('throws when username already exists', function () {

    // -------- Mock Repository --------
    $repo = $this->createMock(UserRepositoryInterface::class);

    // findByUsername → 既存ユーザーを返す
    $repo->method('findByUsername')
        ->willReturn(
            new User(
                username: 'alice',
                email: 'old@example.com',
                passwordHash: 'hashed_password'
            )
        );

    $useCase = new UserRegisterUseCase($repo);

    $input = new UserRegisterInput(
        username: 'alice',
        email: 'alice@example.com',
        plainPassword: 'secret123'
    );

    $this->expectException(ApplicationException::class);

    $useCase->execute($input);
});
