<?php

declare(strict_types=1);

namespace Infrastructure\Db\User;

use PDO;
use Domain\Entities\User;
use Domain\Repositories\UserRepositoryInterface;

final class PdoUserRepository implements UserRepositoryInterface
{
    public function __construct(
        private PDO $pdo
    ) {}

    /**
     * ユーザー名で検索
     */
    public function findByUsername(string $username): ?User
    {
        $sql = 'SELECT username, email, password FROM users WHERE username = :username LIMIT 1';

        $stmt = $this->pdo->prepare($sql);
        $stmt->bindValue(':username', $username, PDO::PARAM_STR);
        $stmt->execute();

        /** @var array{username: string, email: string, password: string}|false $row */
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$row) {
            return null;
        }

        return new User(
            username: $row['username'],
            email: $row['email'],
            passwordHash: $row['password']
        );
    }

    /**
     * 新規ユーザー登録
     */
    public function save(User $user): User
    {
        $sql = 'INSERT INTO users (username, email, password) 
                VALUES (:username, :email, :password)';

        $stmt = $this->pdo->prepare($sql);

        $stmt->bindValue(':username', $user->username(), PDO::PARAM_STR);
        $stmt->bindValue(':email', $user->email(), PDO::PARAM_STR);
        $stmt->bindValue(':password', $user->passwordHash(), PDO::PARAM_STR);

        $stmt->execute();

        // 保存後の最新 User エンティティを返す
        // (将来 id などが追加された場合に備えて新しいインスタンスを返す)
        return new User(
            username: $user->username(),
            email: $user->email(),
            passwordHash: $user->passwordHash()
        );
    }
}
