<?php

declare(strict_types=1);

namespace Infrastructure\Bootstrap;

use PDO;
use Domain\Repositories\UserRepositoryInterface;
use Application\UseCases\User\UserRegisterUseCase;
use Application\UseCases\User\UserRegisterUseCaseInterface;
use Adapter\Controllers\UserController;
use Infrastructure\Http\Router;
use Infrastructure\Db\User\PdoUserRepository;

/** 
 * @param array{
 *   driver?: string,
 *   dsn: string,
 *   user: string|null,
 *   pass: string|null,
 * } $config
 */
function build_pdo(array $config): PDO
{
    return new PDO(
        $config['dsn'],
        $config['user'],
        $config['pass'],
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
}

function build_user_repository(PDO $pdo): UserRepositoryInterface
{
    return new PdoUserRepository($pdo);
}

function build_user_register_usecase(UserRepositoryInterface $repo): UserRegisterUseCaseInterface
{
    return new UserRegisterUseCase($repo);
}

function build_user_controller(UserRegisterUseCaseInterface $uc): UserController
{
    return new UserController($uc);
}

/**
 * ルーターまで含めて「全部組み立てて返す」コンポジション
 *
 * @param array{
 *     db: array{
 *         driver?: string,
 *         dsn: string,
 *         user: string|null,
 *         pass: string|null,
 *     }
 * } $config
 */
function build_router(array $config): Router
{
    $pdo            = build_pdo($config['db']);
    $userRepository = build_user_repository($pdo);
    $registerUc     = build_user_register_usecase($userRepository);
    $userController = build_user_controller($registerUc);

    $routes = [
        'POST register' => [$userController, 'register'],
    ];

    return new Router($routes);
}
