<?php

declare(strict_types=1);

namespace Domain\Entities;

use InvalidArgumentException;

/**
 * User エンティティ（最小構成）
 * username, email, passwordHash の整合性のみ保持
 */
final class User
{
    public function __construct(
        private string $username,
        private string $email,
        private string $passwordHash
    ) {
        $this->assertValid();
    }

    /**
     * 新規作成用ファクトリメソッド
     */
    public static function create(
        string $username,
        string $email,
        string $passwordHash
    ): self {
        return new self(
            username: $username,
            email: $email,
            passwordHash: $passwordHash
        );
    }

    // ====================
    // Getter
    // ====================

    public function username(): string { return $this->username; }

    public function email(): string { return $this->email; }

    public function passwordHash(): string { return $this->passwordHash; }

    // ====================
    // Validation
    // ====================

    private function assertValid(): void
    {
        if ($this->username === '') {
            throw new InvalidArgumentException('ユーザー名がありません。');
        }

        if ($this->email === '') {
            throw new InvalidArgumentException('emailがありません。');
        }

        if (!filter_var($this->email, FILTER_VALIDATE_EMAIL)) {
            throw new InvalidArgumentException('メールアドレスの形式が不正です。');
        }

        if ($this->passwordHash === '') {
            throw new InvalidArgumentException('パスワードがありません。');
        }
    }
}
