<?php

declare(strict_types=1);

namespace Application\UseCases\User;

use Domain\Entities\User;
use Domain\Repositories\UserRepositoryInterface;
use Application\Exception\ApplicationException;

/**
 * ユーザー登録ユースケース
 *
 * - 重複ユーザー名チェック
 * - パスワードハッシュ
 * - User エンティティ生成
 * - リポジトリへ保存
 *
 * フラッシュメッセージやレスポンス生成は行わず、
 * 例外や Output を返すところまでが責務。
 */
final class UserRegisterUseCase implements UserRegisterUseCaseInterface
{
    public function __construct(
        private UserRepositoryInterface $userRepository
    ) {}

    public function execute(UserRegisterInput $input): UserRegisterOutput
    {
        // 1. ユーザー名の重複チェック
        $existing = $this->userRepository->findByUsername($input->username());
        if ($existing !== null) {
            // Controller など外側の層で catch してフラッシュメッセージに変換する
            throw new ApplicationException('そのユーザー名はすでに使われています。');
        }

        // 2. パスワードハッシュ（ドメインでは平文を持たない）
        $passwordHash = password_hash($input->plainPassword(), PASSWORD_DEFAULT);

        // 3. User エンティティ生成
        $user = User::create(
            username: $input->username(),
            email: $input->email(),
            passwordHash: $passwordHash,
        );

        // 4. リポジトリ経由で永続化
        $saved = $this->userRepository->save($user);

        // 5. 出力モデルに詰めて返す
        return new UserRegisterOutput(
            username: $saved->username(),
        );
    }
}
