<?php

declare(strict_types=1);

namespace Adapter\Controllers;

use Application\UseCases\User\UserRegisterUseCaseInterface;
use Application\UseCases\User\UserRegisterInput;
use Application\UseCases\User\UserRegisterOutput;
use Application\Exception\ApplicationException;
use models\UserSchema;
use utils\Flush;
use InvalidArgumentException;
use Throwable;

final class UserController
{
    public function __construct(
        private UserRegisterUseCaseInterface $registerUseCase
    ) {}

    public function register(): void
    {
        try {
            // 1. リクエスト入力を取得（UI層の責務）
            $input = new UserRegisterInput(
                username: get_param('username', ''),
                email: get_param('email', ''),
                plainPassword: get_param('password', ''),
            );

            // 2. ユースケース呼び出し
            $output = $this->registerUseCase->execute($input);

            // 3. ログイン状態にする（UI/Infraの責務）
            $this->login($output);

            // 4. フラッシュ＋リダイレクト（UI 層の責務）
            Flush::push(Flush::INFO, 'Yelp Campへようこそ！');
            redirect('/campgrounds');
            return;

            // 業務的エラー（重複ユーザーなど）
        } catch (ApplicationException | InvalidArgumentException $e) {
            Flush::push(Flush::ERROR, $e->getMessage());
            redirect('/register');
            return;

            // 予期しないエラー（DB障害など）
        } catch (Throwable $e) {
            Flush::push(Flush::ERROR, 'エラーが発生しました');
            redirect('/register');
            return;
        }
    }

    /**
     * UserRegisterOutput を元に簡易ログインを行う
     */
    private function login(UserRegisterOutput $data): void
    {
        // 既存 UserSchema を再構築（最低限の項目だけセット）
        $schema = new UserSchema();
        $schema->username = $data->username();

        // 既存 Session 機能を使ってログイン状態にする
        UserSchema::setSession($schema);
    }
}
