<?php

namespace models;

use InvalidArgumentException;
use models\ImageSchema;
use utils\YelpCampError;

class CampGroundSchema
{
    public int $id;
    public string $title;
    public int $price;
    public string $description;
    public string $location;
    /**
     * @var array<ImageSchema>
     */
    public array $images;
    public string $author; // = users.username
    public string $geometry;
    /**
     * @var array<string>
     */
    public array $deleteImages;

    /**
     * @param object $obj
     */
    public static function cast(object $obj): self
    {
        if (!($obj instanceof self)) {
            $name = get_class($obj);
            throw new InvalidArgumentException("{$name} は CampGroundSchema ではありません。");
        }
        return $obj;
    }

    public static function getModel(): self
    {
        $params = get_asoc_params('campground', '');
        $title = get_item($params, 'title');
        $location = get_item($params, 'location');
        $price = get_item($params, 'price');
        $description = get_item($params, 'description');

        if (empty($title)) throw new YelpCampError('キャンプ名がありません。');
        if (empty($location)) throw new YelpCampError('住所がありません。');
        if (empty($price)) throw new YelpCampError('値段がありません。');
        if (!is_numeric($price)) throw new YelpCampError('値段が数値ではありません。');
        if (intval($price) < 0) throw new YelpCampError('値段がマイナスになっています。');
        if (empty($description)) throw new YelpCampError('説明が入力されていません。');

        $camp = new CampGroundSchema;
        $camp->title = $title;
        $camp->location = $location;
        $camp->price = intval($price);
        $camp->description = $description;
        $camp->images = self::getImages();

        $deleteImages = get_params('deleteImages', '');
        $camp->deleteImages = toStrArray($deleteImages);

        return $camp;
    }

    public function isOwner(UserSchema|null $user): bool
    {
        return !empty($user) && $user->username === $this->author;
    }

    /**
     * @param array<string> $images
     */
    public function addImages(array $images): void
    {
        if (empty($this->images)) $this->images = [];

        foreach ($images as $image) {
            $img = new ImageSchema;
            $img->filename = $image;

            $fname = pathinfo($image, PATHINFO_FILENAME);
            $ext = pathinfo($image, PATHINFO_EXTENSION);
            $thumb_name = self::getUniqueFname("../uploads/thumbnail/{$fname}.{$ext}") ?? '';

            $img->thumbnail = self::createThubnail("../{$image}", $thumb_name) ? "uploads/thumbnail/{$fname}.{$ext}" : $image;
            array_push($this->images, $img);
        }
    }

    /**
     * @return array{geometry: mixed, title: string, location: string, properties: array{popupMarkup: string}}
     */
    public function getGeoData(): array
    {
        $href = BASE_CONTEXT_PATH . "campgrounds/{$this->id}";
        $summary = mb_substr($this->description, 0, 20);
        if (mb_strlen($this->description) > 20) $summary .= "...";
        $geo_info = [
            'geometry' => json_decode($this->geometry),
            'title' => $this->title,
            'location' => $this->location,
            'properties' => ['popupMarkup' => "<strong><a href=\"{$href}\">{$this->title}</a></strong>\n<p>{$summary}</p>"]
        ];

        return $geo_info;
    }

    private static function getUniqueFname(string $filename): string|null
    {
        $dname = pathinfo($filename, PATHINFO_DIRNAME);
        $fname = pathinfo($filename, PATHINFO_FILENAME);
        $ext = pathinfo($filename, PATHINFO_EXTENSION);

        $counter = 1;
        $temp_fname = $fname;

        $mask = umask(002);
        try {
            if (!is_dir($dname) && !mkdir($dname, 0775, true)) return null;
        } finally {
            umask($mask);
        }

        while (file_exists("{$dname}/{$temp_fname}.{$ext}")) {
            $temp_fname = "{$fname}-{$counter}";
            $counter += 1;
        }

        return "{$dname}/{$temp_fname}.{$ext}";
    }

    /**
     * @return array<ImageSchema>
     */
    private static function getImages()
    {
        $results = [];

        if (!isset($_FILES['image'])) return [];

        // @phpstan-ignore offsetAccess.nonOffsetAccessible
        $files = $_FILES['image']['error'];

        if (is_iterable($files)) {
            foreach ($files as $key => $error) {
                if ($error) continue;
                $fname = basename(toString($_FILES['image']['name'][$key]));
                $save_name = self::getUniqueFname('uploads/' . $fname);
                if (empty($save_name)) continue;
                if (!move_uploaded_file(toString($_FILES['image']['tmp_name'][$key]), $save_name)) continue;
                $image = new ImageSchema;
                $image->filename = $save_name;
                $image->thumbnail = self::createThubnail2($save_name);
                array_push($results, $image);
            }
        }

        return $results;
    }

    /**
     * need [sudo apt install imagemagick]
     */
    private static function createThubnail(string $save_name, string $thumb_name): string
    {
        $command = IMAGICK . " {$save_name} -thumbnail 100x100 {$thumb_name}";
        exec($command, $out, $code);
        return $code === 0 ? $thumb_name : '';
    }

    /**
     * need [sudo apt install imagemagick]
     */
    private static function createThubnail2(string $save_name): string
    {
        $fname = pathinfo($save_name, PATHINFO_FILENAME);
        $ext = pathinfo($save_name, PATHINFO_EXTENSION);
        $thumb_name = self::getUniqueFname("uploads/thumbnail/{$fname}.{$ext}") ?? '';

        if (empty($thumb_name)) return $save_name;

        $command = IMAGICK . " {$save_name} -thumbnail 100x100 {$thumb_name}";
        exec($command, $out, $code);
        return $code === 0 ? $thumb_name : $save_name;
    }
}

define('IMAGICK', IS_WINDOWS ? 'magick' : 'convert');
