<?php

declare(strict_types=1);

require_once 'vendor/autoload.php';

/**
 * アプリケーション設定
 *
 * db.driver は将来的なデータベース実装切り替え用の識別子です。
 * （本記事では PDO 固定で使用します）
 *
 * @var array{
 *   db: array{
 *     driver?: string,
 *     dsn: string,
 *     user: string|null,
 *     pass: string|null,
 *   }
 * } $config
 */
$config = require __DIR__ . '/config.php';

require_once 'utils/session.php';
require_once 'utils/flush.php';
require_once 'utils/helper.php';
require_once 'utils/returnTo.php';
require_once 'utils/middleware.php';
require_once 'utils/YelpCampError.php';
require_once 'utils/NotFoundException.php';

require_once 'models/user.php';
require_once 'models/images.php';
require_once 'models/review.php';
require_once 'models/campground.php';

require_once 'db/datasource.php';
require_once 'db/users.query.php';
require_once 'db/images.query.php';
require_once 'db/reviews.query.php';
require_once 'db/campgrounds.query.php';

require_once 'views/partials/flush.php';
require_once 'views/partials/navbar.php';
require_once 'views/partials/footer.php';
require_once 'views/layouts/boilerplate.php';

require_once 'views/404.php';
require_once 'views/error.php';
require_once 'views/users/login.php';
require_once 'views/users/register.php';
require_once 'views/campgrounds/new.php';
require_once 'views/campgrounds/edit.php';
require_once 'views/campgrounds/show.php';
require_once 'views/campgrounds/index.php';

require_once 'controllers/users.php';
require_once 'controllers/reviews.php';
require_once 'controllers/campgrounds.php';

require_once 'routes/users.php';
require_once 'routes/reviews.php';
require_once 'routes/campgrounds.php';

require_once 'src/infrastructure/Bootstrap/container.php';

use utils\YelpCampError;
use utils\NotFoundException;

use Infrastructure\Bootstrap;

session_start();

$router = Bootstrap\build_router($config);

try {

    ['method' => $method, 'path' => $rpath] = get_method();

    switch (true) {

        case $rpath === '':
            require_once 'views/home.php';
            break;

        case \routes\users($method, $rpath, $router):
            break;

        case \routes\reviews($method, $rpath):
            break;

        case \routes\campgrounds($method, $rpath):
            break;

        default:
            throw new NotFoundException();
    }
} catch (YelpCampError $e) {
    \view\error\index($e);
    http_response_code($e->getCode());
    if (!empty($e->redirect_url)) redirect($e->redirect_url);
} catch (Error $e) {
    \view\error\index($e);
    http_response_code($e->getCode());
} catch (NotFoundException $e) {
    \view\not_found\index();
    http_response_code(404);
} catch (Throwable $e) {
    \view\error\index($e);
    die();
}
